/**
 * @file	halopus.h
 * @brief	OPUS hardware abstraction library (libhalopus) generic header
 *
 * @author	Thomas Pinske <tpinske@topcon.com>
 * @copyright	( c ) 2016 Wachendorff Elektronik GmbH & Co. KG, Geisenheim
 * 		All rights reserved.
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License
 *  as published by the Free Software Foundation; either version 2
 *  of the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */

#ifndef _HALOPUS_H_
#define _HALOPUS_H_

#if defined _WIN32 || defined __CYGWIN__
	#ifdef BUILDING_DLL
		#ifdef __GNUC__
			#define OPUS_DLL_EXPORT __attribute__ ((dllexport))
		#else
			#define OPUS_DLL_EXPORT __declspec(dllexport)
		#endif
	#else
		#ifdef __GNUC__
			#define OPUS_DLL_EXPORT __attribute__ ((dllimport))
		#else
			#define OPUS_DLL_EXPORT __declspec(dllimport)
		#endif
	#endif
	#define OPUS_DLL_LOCAL
#else
	#if __GNUC__ >= 4
		#define OPUS_DLL_EXPORT __attribute__ ((visibility ("default")))
		#define OPUS_DLL_LOCAL  __attribute__ ((visibility ("hidden")))
	#else
		#define OPUS_DLL_EXPORT
		#define OPUS_DLL_LOCAL
	#endif
#endif

/* do not include this directly in your project! */
#include <halopus_platform.h>

#include <sys/types.h>

/******************************************************************************
 * ERROR HANDLING
 *****************************************************************************/
/**
 * @enum	opus_errors
 * @brief	return codes from library functions
 */
enum opus_errors {
	OPUS_ERROR_SUCCESS = 0,		/*!< operation succeeded */

	/* error codes */
	OPUS_ERROR_SYS = -1,		/*!< system error, check variable errno for error code */
	OPUS_ERROR_CONV = -2,		/*!< error during conversion of data */
	OPUS_ERROR_INVALID = -3,	/*!< invalid value or parameter */
	OPUS_ERROR_NOT_FOUND = -4,	/*!< element not found */
	OPUS_ERROR_CRITICAL_MEM = -5,	/*!< critical memory condition */
	OPUS_ERROR_BROKEN_DATA = -6,	/*!< read data is broken / NOK */
	OPUS_ERROR_PROTO_ERROR = -7,	/*!< protocol error */
	OPUS_ERROR_TIMEOUT = -8,	/*!< operation timed out */
};

#define opus_is_error(_err)		(_err < OPUS_ERROR_SUCCESS)


/******************************************************************************
 * SENSORS
 *****************************************************************************/
OPUS_DLL_EXPORT extern int opus_sensor_get_value(enum opus_sensors sensor, int32_t *value);


/******************************************************************************
 * POWER REGULATORS
 *****************************************************************************/
/**
 * @enum	opus_regulator_states
 * @brief	power regulator states
 */
enum opus_regulator_states {
	OPUS_REGULATOR_OFF = 0,			/*!< power regulator is off */
	OPUS_REGULATOR_ON = 1,			/*!< power regulator is on */
};

/**
 * @enum	opus_regulator_event_types
 * @brief	power regulator event types
 */
enum opus_regulator_event_types {
	OPUS_REGULATOR_EVENT_UNDEF = 0,		/*!< invalid event */
	OPUS_REGULATOR_EVENT_ENABLE,		/*!< power regulator was switched on */
	OPUS_REGULATOR_EVENT_DISABLE,		/*!< power regulator was switched off */
};

/**
 * @struct	opus_regulator_event
 * @brief	describes a single power regulator event
 */
struct opus_regulator_event {
	u_int32_t id;				/*!< regulator event id,
						     retrieve with @fn opus_regulator_get_event_id */
	enum opus_regulator_event_types type;	/*!< event type */
};

OPUS_DLL_EXPORT extern int opus_regulator_get_path(const char *name, char *path, size_t size);
OPUS_DLL_EXPORT extern int opus_regulator_get_state(const char *name);
OPUS_DLL_EXPORT extern int opus_regulator_get_event_id(const char *name, u_int32_t *id);
OPUS_DLL_EXPORT extern int opus_regulator_open_event_notifier(u_int16_t *notify_id);
OPUS_DLL_EXPORT extern int opus_regulator_close_event_notifier(int fd);
OPUS_DLL_EXPORT extern int opus_regulator_wait_event(int fd, u_int16_t notify_id,
		struct opus_regulator_event *event, u_int32_t timeout_us);


/******************************************************************************
 * STUR
 *****************************************************************************/
OPUS_DLL_EXPORT extern int opus_stur_execute(const char *zone, size_t crit_order,
	size_t crit_blocks, size_t crit_pages, int auto_correct);

#endif /* _HALOPUS_H_ */
