/**
 * @file	wepmd_api.h
 * @brief	API definitions for the Wachendorff Power Management Daemon (wepmd)
 *
 * @author	Thomas Pinske <tpinske@topcon.com>
 * @author	Markus Kalin <mkalin@topcon.com>
 * @copyright	( c ) 2010 - 2015 Wachendorff Elektronik GmbH & Co. KG, Geisenheim
 * 		All rights reserved.
 *
 * This file contains the definitions for the WE Power Management Daemon
 * application programming interface. Include this to your own sources, if
 * you want to use this API.
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License
 *  as published by the Free Software Foundation; either version 2
 *  of the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */

#ifndef WEPMD_API_H_
#define WEPMD_API_H_

/*
 * path of the API socket (UNIX domain socket)
 */
#define WEPMD_API_SOCKET    "/var/run/wepmd_api"


#define WEPMD_API_VERSION 4


/**
 * @enum wepmd_pwr_mode_t
 * @brief Defines all power modes of the machine
 */
typedef enum
{
	WEPMD_PWR_MODE_ON = 0,     	// 0
	WEPMD_PWR_MODE_LOW_POWER,	// 1
	WEPMD_PWR_MODE_SLEEP,		// 2
	WEPMD_PWR_MODE_OFF,			// 3
	WEPMD_PWR_MODE_SILENT_ON,	// 4

	/* last entry, also number of power modes */
	WEPMD_PWR_MODE_COUNT
} wepmd_pwr_mode_t;


/**
 * @enum wepmd_dev_state_t
 * @brief Defines the suspend states of a device
 */
typedef enum
{
  WEPMD_DEV_STATE_OFF = 0,
  WEPMD_DEV_STATE_ON,
} wepmd_dev_state_t;


/**
 * @struct wepmd_timeout_t
 * @brief Power mode timout settings
 */
typedef struct
{
  /* identifies the corresponding power mode for the timeout */
  wepmd_pwr_mode_t mode;
  
  /* timeout value in s */
  unsigned int timeout;
} wepmd_timeout_t;


/**
 * @struct wepmd_pwr_t
 * @brief Power mode settings
 */
typedef struct
{
  /* defines the power mode */
  wepmd_pwr_mode_t mode;
} wepmd_pwr_t;


/*
 * maximum length of device name. Use this for string copy function etc.
 */
#define WEPMD_MAX_DEV_NAME  15


/**
 * @struct wepmd_device_t
 * @brief Device suspend state settings
 */
typedef struct
{
  /* name of the device (can be retrieved over device list command) */
  char name[WEPMD_MAX_DEV_NAME + 1];
  
  /* suspend state of the device during low-power mode */
  wepmd_dev_state_t state;
} wepmd_device_t;


/**
 * @struct wepmd_devicelist_t
 * @brief List entry for device list command
 * 
 * The device list is transfered as a sequence of messages, where each message
 * contains one device. The end of the list is marked with a special marker.
 * This structure is used to transfer one message.
 */ 
typedef struct
{
  /* Name of the device (end marker contains empty string) */
  char name[WEPMD_MAX_DEV_NAME + 1];
  
  /* A sequence number, starting at zero. Can be used to check, if all list
   * members were received. (end marker contains number of devices transfered) */
  int sequence;
  
  /* This field marks the end of the list (0 = device, 1 = end marker). The
   * last transfered entry doesn't hold a valid device. It's only used as
   * end marker, containing this flag and the numer of devoces transfered. */
  int end;
} wepmd_devicelist_t;


/**
 * @enum wepmd_cmd_t
 * @brief List of all API commands
 * Append new commands at the end to ensure backward compatibility!
 */
typedef enum
{
  WEPMD_CMD_GET_TIMEOUT = 0,
  WEPMD_CMD_SET_TIMEOUT,
  WEPMD_CMD_TIMEOUT,
  WEPMD_CMD_GET_DEVICE,
  WEPMD_CMD_SET_DEVICE,
  WEPMD_CMD_DEVICE,
  WEPMD_CMD_GET_DEVICELIST,
  WEPMD_CMD_DEVICELIST,
  WEPMD_CMD_GET_MODE,
  WEPMD_CMD_SET_MODE,
  WEPMD_CMD_MODE,
  WEPMD_CMD_MODE_CHANGE,
  WEPMD_CMD_REBOOT,
  WEPMD_CMD_HALT,
  WEPMD_CMD_GET_PROPAGATION_TIME,
  WEPMD_CMD_SET_PROPAGATION_TIME,
  WEPMD_CMD_PROPAGATION_TIME,
  WEPMD_CMD_MODE_PROPAGATE,
  WEPMD_CMD_RETARD_MODE,
  WEPMD_CMD_GET_API_VERSION,
  WEPMD_CMD_API_VERSION,
  WEPMD_CMD_ERROR,
} wepmd_cmd_t;


/**
 * @enum wepmd_err_type_t
 * @brief List of all API errors
 * Append new errors at the end to ensure backward compatibility!
 */
typedef enum
{
  WEPMD_ERR_UNKNOWN_COMMAND = 0,
  WEPMD_ERR_NOT_A_CLIENT_COMMAND,
  WEPMD_ERR_ALREADY_SET,
  WEPMD_ERR_NOT_SUPPORTED,
  WEPMD_ERR_NO_DEVICE,
} wepmd_err_type_t;


/**
 * @struct wepmd_error_t
 * @brief error messages for socket communication
 */
typedef struct
{
  wepmd_cmd_t      command;             /* command which caused the error */
  wepmd_err_type_t errorType;           /* type of error */
} wepmd_error_t;


/**
 * @union wepmd_msg_data_t
 * @brief Payload data of one API message
 * 
 * This is the data portion of one API message. Each message exists of a
 * command the corresponding payload data, which can be different from
 * command to command. Please take a look to the documentation for further
 * details of the API message format.
 */
typedef union
{
  wepmd_timeout_t timeout;              /* used for timeout settings */
  wepmd_device_t device;                /* used for device suspend state settings */
  wepmd_pwr_t power;                    /* used for power mode settings */
  wepmd_devicelist_t list;              /* used for device list transfers */
  unsigned int apiVersion;              /* used to return API version */
  wepmd_error_t error;                  /* used for error messages */
} wepmd_msg_data_t;


/**
 * @struct wepmd_msg_t
 * @brief wepmd API message
 */
typedef struct
{
  /* API command */
  wepmd_cmd_t command;
  
  /* payload data */
  wepmd_msg_data_t data;
} wepmd_msg_t;


#endif /*WEPMD_API_H_*/
